﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

using Microsoft.EntityFrameworkCore;
using Microsoft.AspNetCore.Identity;

using WebFileServ.Model.Entities.Identity;
using WebFileServ.Model.Entities.FS;

using WebFileServ.Model.DAL.DataBase.EF;

namespace WebFileServ.Model.DAL.DataInit
{

    /// <summary>
    /// Класс для управления состоянием БД
    /// </summary>
    public class DataInitializer
        : IDisposable
    {
        public DataInitializerConfig DataInitializerConfig { private set; get; }

        private readonly ApplicationDbContext Context;

        private readonly UserManager<ApplicationUser> UserManager;
        private readonly RoleManager<ApplicationRole> RoleManager;



        
        public DataInitializer
            (
            DataInitializerConfig dataInitializerConfig,

            ApplicationDbContext context,

            UserManager<ApplicationUser> userManager,
            RoleManager<ApplicationRole> roleManager
            )
        {
            DataInitializerConfig = dataInitializerConfig;

            Context = context;

            UserManager = userManager;
            RoleManager = roleManager;
        }



        public async Task Init()
        {
            if (DataInitializerConfig.ReCreateDataBase)
            {
                await RemoveDbIfExsist();
            }

            if (DataInitializerConfig.AutoApplyMigration)
            {
                ApplyMigrations();
            }

            if (DataInitializerConfig.FillData)
            {
                await CreateAdminUserIfNotExsist();
                await CreateTestFsItem();
            }            
        }


        /// <summary>
        /// Удаление существующей базы. Для тестов.
        /// </summary>
        private async Task RemoveDbIfExsist()
        {           
            await Context.Database.EnsureDeletedAsync();
        }

        /// <summary>
        /// Создание БД если нет и автоматическое накатывание мигарций
        /// </summary>
        private void ApplyMigrations()
        {
            Context.Database.Migrate();
        }



        /// <summary>
        /// Иницилизация пользователя администратора по-умолчанию.
        /// Иницилизация роли администраторов
        /// </summary>
        /// <returns></returns>
        private async Task CreateAdminUserIfNotExsist()
        {
            var email = "admin@admin";
            var password = "demo";

            var roleName = "AdminRole";


            var adminRole = await RoleManager
                .FindByNameAsync(roleName);

            if (adminRole == null)
            {
                var createRoleResult = await RoleManager.CreateAsync(
                    new ApplicationRole(roleName)
                    );

                if (!createRoleResult.Succeeded)
                {
                    throw new Exception(
                        createRoleResult.Errors.FirstOrDefault()?.Description
                        );
                }
            }


            var adminUser = await UserManager
                .FindByNameAsync(email);

            if (adminUser == null)
            {
                var createUserResult = await UserManager
                    .CreateAsync(
                        new ApplicationUser()
                        {
                            Email = email,
                            UserName = email
                        },
                        password
                    );

                if (!createUserResult.Succeeded)
                {
                    throw new Exception(
                        createUserResult.Errors.FirstOrDefault()?.Description
                        );
                }

                adminUser = await UserManager
                    .FindByNameAsync(email);

                var addToRoleResult = await UserManager
                    .AddToRoleAsync(adminUser, roleName);

                if (!addToRoleResult.Succeeded)
                {
                    throw new Exception(
                        addToRoleResult.Errors.FirstOrDefault()?.Description
                        );
                }

                var adminRoles = adminUser
                    .RolesDict;
            }
        }


        private async Task CreateTestFsItem() 
        {
            var adminUser = Context.Users
                .FirstOrDefault();

            var rootDirectory = Context.FsRootDirectories
                .Add(
                    new Fs_RootDirectory() 
                    {
                        //OwnerUser = adminUser
                    }
                    )
                .Entity;

            await Context
                .SaveChangesAsync();

            var roleName = "AdminRole";
            var role = await Context.Roles
                .FirstOrDefaultAsync(e => e.Name == roleName);

            if (role == null)
            {
                throw new Exception("Role not found");
            }


            Context.FsItemGroupPermissions.Add(
                new FsItemGroupPermission() 
                {
                    Role = role,
                    RootDirectory = rootDirectory,
                    Permission = EnumPermission.Read
                }
                );

            //rootDirectory.GroupPermission = new List<FsItemGroupPermission>() 
            //{
            //    new FsItemGroupPermission()
            //    {
            //        Role = role,
            //        RootDirectory = rootDirectory,
            //        Permission = EnumPermission.Read
            //    }
            //};
            await Context
                .SaveChangesAsync();

            var permission = rootDirectory
                .GroupPermission
                .FirstOrDefault(e => e.RoleId == role.Id);

            if (permission == null)
            {
                throw new Exception("Нет доступа");
            }

            Console.WriteLine($"Ваш доступ:{permission.Permission}");
        }



        #region Dispose

        private bool disposed = false;

        // реализация интерфейса IDisposable.
        public void Dispose()
        {
            Dispose(true);
            // подавляем финализацию
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!disposed)
            {
                if (disposing)
                {
                    // Освобождаем управляемые ресурсы
                    Context.Dispose();

                    UserManager.Dispose();
                    RoleManager.Dispose();
                }
                // освобождаем неуправляемые объекты
                disposed = true;
            }
        }

        // Деструктор
        ~DataInitializer()
        {
            Dispose(false);
        }

        #endregion

    }
}
