﻿using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using System.Linq;
using System.IO;
using System.Diagnostics;

using Microsoft.EntityFrameworkCore;

using Tools.PdfProvider;

using LibraryText;
using LibraryText.Entities;

using Calibre.Model.Database;
using Calibre.Model.Database.Entities;
using Calibre.Model.Domain.Entities;
using Calibre.Model.Domain.Dal;

namespace Calibre
{
    class Program
    {
        static async Task Main(string[] args)
        {
            //var connectionString = @"";

            //using (var context = new CalibreContext(connectionString))
            //{
            //    var books = context.Books
            //        .OrderBy(e => e.Id)
            //       .ToArray();
            //}

            //await SearchByTitleInDirectory(
            //    new DirectoryInfo(@"S:\BooksText\Calibre"),
            //    "Python Guide"
            //    );


            //await SearchInPdfText(
            //    new DirectoryInfo(@"S:\BooksText\Calibre"),
            //    "NumPy"
            //    );

            await LoadToDb(
                new DirectoryInfo(@"S:\BooksText\Calibre")
                );
        }


        static async Task<BookWithLibraryContainer[]> SearchByTitleInDirectory(
            DirectoryInfo directory,
            string containsTitle,
            bool printLine = true
            )
        {
            var watch = Stopwatch.StartNew();
            var searchProvider = new SearchProvider(
                new SimplePdfReader()
                );

            var libraries = Library.SearchAll(directory);
            var searchResult = await searchProvider.GetBooksByLibrariesAsync(
                libraries: libraries
                )
                .ConfigureAwait(false);
            watch.Stop();


            var allBooks = searchResult
                .SelectMany(e => e.Value)
                .ToArray();

            var allTags = allBooks
                .SelectMany(e => e.Book.Tags)
                .Select(e => e.TagItem)
                .Distinct()
                .ToHashSet();

            var result = allBooks;

            if (!string.IsNullOrEmpty(containsTitle))
            {
                result = allBooks
                .Where(
                    e =>
                        //Заголовок
                        e.Book.Title.Contains(containsTitle, StringComparison.OrdinalIgnoreCase)
                        //Теги
                        || e.Book.Tags.Any(
                                e => e.TagItem.Name.Contains(containsTitle, StringComparison.OrdinalIgnoreCase)
                                )
                    )
                .ToArray();
            }

            if (printLine)
            {
                foreach (var elem in result)
                {
                    Console.WriteLine($"{elem.Library.Data.Name}|{elem.Book.Title}|{elem.Book.AuthorSort}");
                }
            }

            return result;
        }


        static async Task<Dictionary<BookWithLibraryContainer, string>> SearchInPdfText(
            DirectoryInfo directory,
            string containsText = null,
            string containsTitle = null,
            bool printLine = true
            )
        {
            var searchProvider = new SearchProvider(
                new SimplePdfReader()
                );

            var books = await SearchByTitleInDirectory(directory, containsTitle, false);
            books = books
                .Where(
                    e => e.Book.FileData
                        .Any(
                            e2 => string.Equals(e2.Format, "pdf", StringComparison.OrdinalIgnoreCase)
                            )
                )
                .ToArray();

            var result = await searchProvider.GetTextByBooksAsync(
                books,
                true,
                (item, text) =>
                {
                    if (string.IsNullOrEmpty(containsText))
                    {
                        return true;
                    }
                    return text.Contains(containsText, StringComparison.OrdinalIgnoreCase);
                }
                );

            if (printLine)
            {
                foreach (var elem in result)
                {
                    Console.WriteLine($"{elem.Key.Library.Data.Name}|{elem.Key.Book.Title}|{elem.Key.Book.AuthorSort}");
                }
            }

            return result;
        }


        static async Task LoadToDb(
            DirectoryInfo directory,
            string containsText = null,
            string containsTitle = null,
            bool printLine = true
            )
        {
            var searchProvider = new SearchProvider(
                new SimplePdfReader()
                );

            var books = await SearchByTitleInDirectory(directory, containsTitle, false);
            books = books
                .Where(
                    e => e.Book.FileData
                        .Any(
                            e2 => string.Equals(e2.Format, "pdf", StringComparison.OrdinalIgnoreCase)
                            )
                )
                .ToArray();


            await searchProvider.GetTextByBooksAsync(
                books,
                true,
                (item, text) =>
                {
                    if (!string.IsNullOrEmpty(containsText))
                    {
                        if (!text.Contains(containsText, StringComparison.OrdinalIgnoreCase))
                        {
                            return false;
                        }
                    }
                    using (ApplicationContext context = new ApplicationContext())
                    {

                        context.Books.Add(                                
                            new BookTextData()
                            {
                                Directory = item.Library.Data.FullName,
                                Text = text,
                                Title = item.Book.Title
                            }                                
                            );
                        context.SaveChanges();
                    }

                    if (printLine)
                    {
                        Console.WriteLine($"{item.Library.Data.Name}|{item.Book.Title}|{item.Book.AuthorSort}|{text?.Length}");
                    }

                    return false;
                }
                );
        }
    }
}
