﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Threading;

using Tools.Collections.Concurrent.AsyncBuffer.Store;

namespace Tools.Collections.Concurrent.AsyncBuffer.PriorityWrapper
{

    public class ConcurrentStorePriorityWrapper<T, TPriority>
        : IConcurrentStorePriorityWrapper<T, TPriority>
    {
        public readonly SortedDictionary<TPriority, IConcurrentStore<T>> PriorityStorages;

        private int _Size;
        public int Size
            => _Size;

        public event Action<IConcurrentStore<T>, T> OnItemAdded;
        public event Action<IConcurrentStore<T>> OnEmpty;

        public ConcurrentStorePriorityWrapper(
            SortedDictionary<TPriority, IConcurrentStore<T>> priorityStorages
            ) 
        {
            _Size = priorityStorages.Sum(e => e.Value.Size);
            PriorityStorages = priorityStorages;
        }

        public int Add(T data)
        {
            return Add(data, default);
        }
        public int Add(T data, TPriority priority)
        {
            PriorityStorages[priority].Add(data);
            Interlocked.Increment(ref _Size);
            OnItemAdded?.Invoke(this, data);
            return _Size;
        }

        public bool TryTake(out T data)
        {
            foreach (var elem in PriorityStorages)
            {
                if (elem.Value.TryTake(out data))
                {
                    Interlocked.Decrement(ref _Size);

                    if (_Size == 0)
                    {
                        OnEmpty?.Invoke(this);
                    }

                    return true;
                }
            }

            data = default;
            return false;
        }
        public bool TryTake(out T data, TPriority priority)
        {
            if (PriorityStorages[priority].TryTake(out data))
            {
                Interlocked.Decrement(ref _Size);

                if (_Size == 0)
                {
                    OnEmpty?.Invoke(this);
                }

                return true;
            }

            data = default;
            return false;
        }


        public IEnumerable<T> AsIEnumerable()
        {
            return PriorityStorages
                .Values
                .SelectMany(e => e.AsIEnumerable());
        }

        public void ForEach(Action<T> action)
        {
            foreach (var elem in PriorityStorages)
            {
                elem.Value.ForEach(action);
            }
        }

        public IReadOnlyDictionary<TPriority, IConcurrentStore<T>> AsDictionary() 
        {
            return PriorityStorages
                as IReadOnlyDictionary<TPriority, IConcurrentStore<T>>;
        }      
    }
    
}
